/*
 * Decompiled with CFR 0.152.
 */
package systems.dmx.ldap.repository;

import java.io.IOException;
import java.util.Collections;
import java.util.Hashtable;
import java.util.List;
import java.util.Objects;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.naming.NamingEnumeration;
import javax.naming.NamingException;
import javax.naming.directory.Attribute;
import javax.naming.directory.Attributes;
import javax.naming.directory.BasicAttribute;
import javax.naming.directory.BasicAttributes;
import javax.naming.directory.DirContext;
import javax.naming.directory.ModificationItem;
import javax.naming.directory.SearchControls;
import javax.naming.directory.SearchResult;
import javax.naming.ldap.InitialLdapContext;
import javax.naming.ldap.LdapContext;
import javax.naming.ldap.StartTlsRequest;
import javax.naming.ldap.StartTlsResponse;
import org.apache.commons.lang3.StringUtils;
import org.springframework.security.crypto.password.LdapShaPasswordEncoder;
import systems.dmx.ldap.Configuration;

class JndiDatasource {
    private static final Logger logger = Logger.getLogger(JndiDatasource.class.getName());
    public static final String LDAP_SEARCH_TEMPLATE = "%s=%s,%s";
    private final Configuration.ProtocolType protocolType;
    private final String connectionUrl;
    private final String userFilter;
    private final String userAttribute;
    private final String userBase;
    private final String userMemberGroup;
    private final String groupBase;
    private final String adminUserName;
    private final String adminDn;

    JndiDatasource(Configuration configuration, String adminUserName, String adminDn) {
        this.protocolType = configuration.protocolType;
        this.connectionUrl = configuration.connectionUrl;
        this.userFilter = configuration.userFilter;
        this.userAttribute = configuration.userAttribute;
        this.userBase = configuration.userBase;
        this.userMemberGroup = configuration.userMemberGroup;
        this.groupBase = configuration.groupBase;
        this.adminUserName = adminUserName;
        this.adminDn = adminDn;
    }

    private String encodePassword(String password) {
        return new LdapShaPasswordEncoder().encode((CharSequence)password);
    }

    void createUser(LdapContext ctx, String userName, String password) throws NamingException {
        String entryDN = String.format(LDAP_SEARCH_TEMPLATE, this.userAttribute, userName, this.userBase);
        BasicAttributes entry = this.createUserNameEntry(userName, this.encodePassword(password));
        ctx.createSubcontext(entryDN, (Attributes)entry);
        if (StringUtils.isNotEmpty((CharSequence)this.userMemberGroup)) {
            ModificationItem mi = new ModificationItem(1, new BasicAttribute("member", entryDN));
            try {
                ctx.modifyAttributes(this.userMemberGroup, new ModificationItem[]{mi});
            }
            catch (NamingException ne) {
                ctx.destroySubcontext(entryDN);
                throw ne;
            }
        }
    }

    private BasicAttributes createUserNameEntry(String userName, String encodedPassword) {
        BasicAttribute cn = new BasicAttribute("cn", userName);
        BasicAttribute sn = new BasicAttribute("sn", "deepamehta-ldap");
        BasicAttribute userPassword = new BasicAttribute("userPassword", encodedPassword);
        BasicAttribute oc = new BasicAttribute("objectClass");
        oc.add("top");
        oc.add("person");
        oc.add("organizationalPerson");
        oc.add("inetOrgPerson");
        BasicAttributes entry = new BasicAttributes();
        entry.put(oc);
        entry.put(cn);
        entry.put(sn);
        entry.put(userPassword);
        return entry;
    }

    void checkCredentialsWithLookup(LdapContext ctx, String username, String password) throws NamingException {
        logger.log(Level.INFO, "Checking credentials for user %s", username);
        LdapContext ctx2 = null;
        try {
            String cn = this.lookupUserCn(ctx, username);
            if (cn == null) {
                logger.warning(() -> String.format("User %s not found in LDAP", username));
                throw new NamingException();
            }
            ctx2 = this.connect(cn, password);
            this.closeQuietly(ctx2);
        }
        catch (NamingException ne) {
            try {
                logger.warning(() -> String.format("Provided credentials for user %s were wrong", username));
                throw ne;
            }
            catch (Throwable throwable) {
                this.closeQuietly(ctx2);
                throw throwable;
            }
        }
    }

    void checkCredentials(String username, String password) throws NamingException {
        logger.info(() -> String.format("Checking credentials for user %s", username));
        LdapContext ctx = null;
        try {
            String cn = this.userNameToEntryDn(username);
            ctx = this.connect(cn, password);
            this.closeQuietly(ctx);
        }
        catch (NamingException ne) {
            try {
                logger.warning(() -> String.format("Provided credentials for user %s were wrong", username));
                throw ne;
            }
            catch (Throwable throwable) {
                this.closeQuietly(ctx);
                throw throwable;
            }
        }
    }

    private String lookupUserCn(LdapContext ctx, String uid) throws NamingException {
        String searchFilter = StringUtils.isEmpty((CharSequence)this.userFilter) ? String.format("(%s=%s)", this.userAttribute, uid) : String.format("(&(%s)(%s=%s))", this.userFilter, this.userAttribute, uid);
        logger.info(() -> String.format("Complete filter expression for user lookup: %s", searchFilter));
        SearchControls searchControls = new SearchControls();
        searchControls.setSearchScope(2);
        NamingEnumeration<SearchResult> results = ctx.search(this.userBase, searchFilter, searchControls);
        logger.info(() -> String.format("Search base is: %s", this.userBase));
        if (results.hasMoreElements()) {
            logger.info("Lookup using search filter returned non-empty result");
            SearchResult searchResult = (SearchResult)results.nextElement();
            if (results.hasMoreElements()) {
                throw new RuntimeException("Ambiguity in LDAP CN query: Matched multiple users for the accountName");
            }
            return searchResult.getNameInNamespace();
        }
        logger.warning("Lookup using search filter was empty.");
        return null;
    }

    void changePassword(LdapContext ctx, String username, String password) throws NamingException {
        logger.info(() -> String.format("Changing password for user %s", username));
        this.changePasswordImpl(ctx, username, this.encodePassword(password));
    }

    private void changePasswordImpl(LdapContext ctx, String userName, String password) throws NamingException {
        String entryDN = this.userNameToEntryDn(userName);
        ModificationItem mi = new ModificationItem(2, new BasicAttribute("userPassword", password));
        try {
            ctx.modifyAttributes(entryDN, new ModificationItem[]{mi});
        }
        catch (NamingException ne) {
            logger.log(Level.WARNING, "Attempt to modify userPassword attribute lead to exception", ne);
            throw ne;
        }
    }

    void deleteUser(LdapContext ctx, String user) throws NamingException {
        try {
            ctx.destroySubcontext(this.userNameToEntryDn(user));
        }
        catch (NamingException e) {
            logger.log(Level.SEVERE, String.format("Unable to delete user from LDAP %s", user), e);
            throw e;
        }
        finally {
            this.closeQuietly(ctx);
        }
    }

    private String userNameToEntryDn(String userName) {
        return String.format(LDAP_SEARCH_TEMPLATE, this.userAttribute, userName, this.userBase);
    }

    LdapContext connect(String username, String password) throws NamingException {
        try {
            InitialLdapContext ctx = new InitialLdapContext(this.createEnvironment(username, password), null);
            if (this.protocolType == Configuration.ProtocolType.STARTTLS) {
                logger.info("Attempting TLS negotiation (StartTLS protocol)");
                StartTlsResponse tls = (StartTlsResponse)ctx.extendedOperation(new StartTlsRequest());
                tls.negotiate();
                logger.info("TLS negotiated successfully.");
            }
            logger.info(() -> String.format("Context for user %s usable", username));
            return ctx;
        }
        catch (IOException e) {
            throw new RuntimeException("Could not establish TLS connection. Connecting failed.", e);
        }
    }

    private Hashtable<String, Object> createEnvironment(String username, String password) {
        Hashtable<String, Object> env = new Hashtable<String, Object>();
        env.put("java.naming.security.authentication", "simple");
        env.put("java.naming.provider.url", this.connectionUrl);
        env.put("java.naming.security.principal", username);
        env.put("java.naming.security.credentials", password);
        env.put("java.naming.factory.initial", "com.sun.jndi.ldap.LdapCtxFactory");
        env.put("java.naming.ldap.attributes.binary", "objectSID");
        return env;
    }

    void closeQuietly(LdapContext ctx) {
        if (ctx != null) {
            try {
                ctx.close();
            }
            catch (NamingException ne) {
                logger.log(Level.WARNING, "Exception while closing connection", ne);
            }
        }
    }

    private String groupDn(String groupName) {
        return String.format("cn=%s,%s", groupName, this.groupBase);
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void createGroup(LdapContext ctx, String group, String user, List<String> members) throws NamingException {
        try {
            String groupDn = this.groupDn(group);
            String firstMemberDn = this.resolveUserDn(ctx, user);
            List<String> otherMemberDns = members.stream().map(it -> this.resolveUserDn(ctx, (String)it)).filter(Objects::nonNull).collect(Collectors.toList());
            this.createGroupImpl(ctx, groupDn, firstMemberDn, otherMemberDns);
        }
        finally {
            this.closeQuietly(ctx);
        }
    }

    private void createGroupImpl(LdapContext ctx, String groupDn, String firstMemberDn, List<String> otherMemberDns) throws NamingException {
        logger.info(() -> String.format("Creating group %s with first member %s and %s other members", groupDn, firstMemberDn, otherMemberDns.size()));
        BasicAttribute member = new BasicAttribute("member", firstMemberDn);
        otherMemberDns.forEach(member::add);
        BasicAttribute oc = new BasicAttribute("objectClass");
        oc.add("top");
        oc.add("groupOfNames");
        BasicAttributes entry = new BasicAttributes();
        entry.put(oc);
        entry.put(member);
        try {
            ctx.createSubcontext(groupDn, (Attributes)entry);
        }
        catch (NamingException ne) {
            logger.log(Level.SEVERE, "Unable to create group subcontext", ne);
            throw ne;
        }
    }

    void addMember(LdapContext ctx, String groupDn, String user) throws NamingException {
        String userEntryDn = this.resolveUserDn(ctx, user);
        if (userEntryDn == null) {
            throw new IllegalStateException("User not found");
        }
        try {
            ctx.lookup(groupDn);
        }
        catch (NamingException ne) {
            logger.info(() -> String.format("Group %s does not exist. Attempting to create it.", groupDn));
            this.createGroupImpl(ctx, groupDn, userEntryDn, Collections.emptyList());
        }
        ModificationItem mi = new ModificationItem(1, new BasicAttribute("member", userEntryDn));
        try {
            ctx.modifyAttributes(groupDn, new ModificationItem[]{mi});
        }
        catch (NamingException ne) {
            logger.log(Level.WARNING, "Attempt to modify member attribute lead to exception", ne);
            throw ne;
        }
    }

    void deleteGroup(LdapContext ctx, String group) throws NamingException {
        String groupDn = this.groupDn(group);
        try {
            logger.info(() -> String.format("Trying to delete group %s", groupDn));
            ctx.destroySubcontext(groupDn);
        }
        catch (NamingException ne) {
            logger.log(Level.WARNING, "Attempt to delete group lead to exception", ne);
            throw ne;
        }
        finally {
            this.closeQuietly(ctx);
        }
    }

    /*
     * WARNING - Removed try catching itself - possible behaviour change.
     */
    void removeMember(LdapContext ctx, String group, String user) throws NamingException {
        String groupDn = this.groupDn(group);
        logger.info(() -> String.format("Removing user %s from group %s", user, groupDn));
        try {
            String userDn = this.resolveUserDn(ctx, user);
            if (userDn == null) {
                throw new IllegalStateException("User not found");
            }
            this.removeMemberImpl(ctx, groupDn, userDn);
        }
        finally {
            this.closeQuietly(ctx);
        }
    }

    void maybeDeleteGroup(LdapContext ctx, DirContext groupContext, String groupDn, String userEntryDn) throws NamingException {
        try {
            Attribute a = groupContext.getAttributes("").get("member");
            if (a.size() == 1 && userEntryDn.equals(a.get(0))) {
                logger.info(() -> String.format("Group %s is now empty. Attempting to delete.", groupDn));
                ctx.destroySubcontext(groupDn);
            }
        }
        catch (NamingException ne) {
            logger.warning(() -> String.format("Unable to check membership or delete the group %s", groupDn));
            throw ne;
        }
    }

    private String resolveUserDn(LdapContext ctx, String userName) {
        String userDn = this.userNameToEntryDn(userName);
        try {
            ctx.lookup(userDn);
            return userDn;
        }
        catch (NamingException ne) {
            if (userName.equals(this.adminUserName)) {
                return this.adminDn;
            }
            logger.warning(() -> String.format("Unable to find regular user %s in LDAP. Ignoring", userName));
            return null;
        }
    }

    private void removeMemberImpl(LdapContext ctx, String groupDn, String userEntryDn) throws NamingException {
        DirContext groupContext;
        try {
            groupContext = (DirContext)ctx.lookup(groupDn);
        }
        catch (NamingException ne) {
            logger.log(Level.WARNING, "Unable to look up group lead to exception", ne);
            throw ne;
        }
        this.maybeDeleteGroup(ctx, groupContext, groupDn, userEntryDn);
        ModificationItem mi = new ModificationItem(3, new BasicAttribute("member", userEntryDn));
        try {
            ctx.modifyAttributes(groupDn, new ModificationItem[]{mi});
        }
        catch (NamingException ne) {
            logger.log(Level.WARNING, "Attempt to modify member attribute lead to exception", ne);
            throw ne;
        }
    }
}

